/************************************************************************
 * @file: AudioSilenceDetector.h
 *
 * @description: This header file contains definition for class SilenceDetector.
 * SilenceDetector is a Post Processing utility to detect the silence introduced
 * in PCM data.
 *
 * @authors: Jens Lorenz, jlorenz@de.adit-jv.com 2016
 *           Vijay Palaniswamy, vijay.palaniswamy@in.bosch.com 2016
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#ifndef _ADIT_UTILITY_AUDIOSILENCEDETECTOR_H_
#define _ADIT_UTILITY_AUDIOSILENCEDETECTOR_H_

#include <string>
#include <vector>

#include "AudioTypes.h"
#include "AudioStreaming.h"
#include "Logging.h"

#define DEFAULT_MIN_MEAN_PERCENTAGE  1                /* Mean in Percentage    */
#define MIN_PCM_DATA_REQ             2                /* Time in milli seconds */
#define DIFF_TOLERANCE              20
#define MIN_FADETIME_REQ             8                /* Time in milli seconds */
#define U8_LEVEL_ZERO                (1 << (8  - 1))
#define U16_LEVEL_ZERO               (1 << (16 - 1))
#define U32_LEVEL_ZERO               (1 << (32 - 1))

namespace adit
{

namespace utility
{

namespace audio
{

typedef struct
{
    AudioFormat  format;
    uint32_t     rate;
    uint32_t     channels;
    uint32_t     frames;
} SdStreamInfo;

enum class SdRet
{
    OK = 0,            /**< Operation is success. */
    NOMEM,             /**< Out of memory. */
    UNSUPPORTED,       /**< The given data or file type is not supported. */
    ALREADY,           /**< Operation already in progress or done. */
    FAILURE            /**< Operation is failed.  */
};

class SilenceDetector : public Streaming
{
public:
    SilenceDetector(Streaming& streamingHandle);

    ~SilenceDetector();

    /**
     * This function is called for configuring a silence detector for the specified parameters.
     * This function can be called after creating handle for silence detector to configure channels, rate, format, Fade time
     * This function cannot be called during processing.
     * @param[IN] streamInfo    Specify parameters like channels, rate, format, Fade time.
     * @return OK               if Silence detector is configured successfully
     * @return UNSUPPORTED      if specified parameters are of unsupported formats and zero channels,Fade time not supported
     * @return FAILURE          if Silence detector is not in state  NONE,memory allocation failed
     * @note: Minimum fade time required is 8 milliseconds
     */
    SdRet configure(const SdStreamInfo streamInfo);

    /**
     * This function is called to start Silence detection process.
     * This function should be called after configure/reset.
     * Activate can be called to restart the fade process with same configured inputs.
     */
    void activate();

    /**
     * This function is called to stop silence detection process
     * Reset can be called any time to stop silence detection process
     * @note: After calling reset,Configure API should be called
     */
    void reset();

private:
    enum class SdState
    {
        NONE       = 0x00,
        CONFIGURED = 0x01,
        PROCESSING = 0x02,
        FADING     = 0x04,
        STOPPED    = 0x08,
    };

    uint32_t mFirstActiveSampleDone;
    std::vector<uint32_t> mMean;
    uint8_t* mSilenceBuf;

    uint32_t mChannels;
    uint32_t mRate;
    uint32_t mFrames;
    uint32_t mMinFramesReq;
    uint32_t mNoOfMeanValues;
    uint32_t mFrameSize;
    uint32_t mBytesPerSample;
    uint64_t mLevelZero;
    uint64_t mMinMeanAmplitude;
    AudioFormat mFormat;
    SdState  mState;
    Streaming& mStreamingHandle;

    /**
     * This function is called for Silence detection
     * This function is to be called after configure() API to start process
     * to restart the fade process with same configured inputs for fadetime and direction.
     * @return FAILURE      if invalid input buffer, not in proper state
     * @return OK           if Silence detection process is successful
     */
    SdRet process(const void* buf, const uint32_t frames);

    SdRet processInternal(const void* buf, const uint32_t frames);

    template<typename T>
    SdRet processData(const T* buf, const uint32_t frames);

    SdRet checkForFading();

    void isSilence();

    void addtoMeanlist(const uint32_t mean);

    template<typename M>
    uint32_t calculateMean(const M* buf, const uint32_t frames);

    void startProcessing();

    void stopProcessing();

    void changeState(const SdState newState);

    SdState getState() { return mState; };

    SdRet updateSdPrivateData();

    void defaults();

    SdRet isFormatSupported(const AudioFormat format);

    void clearSdPrivateData();

    template<typename L>
    void setSilenceData(void* buffer, const L level, const uint32_t frames);

    SdRet allocSilenceBuffer();

    void freeInternalBuffers();

    void parseEnvVariable();

    template<typename E>
    void swapEndianess(E& value);

    bool isSwapEndianess(const AudioFormat format);

public:

    /* Inherited API's from Streaming class */
    void error(const std::string& data) const final;

    void warning(const std::string& data) const final;

    void info(const std::string& data) const final;

    void debug(const std::string& data) const final;

    eLogLevel checkLogLevel(void) const final;

    AudioState processing(unsigned char *in, unsigned char **out, uint32_t &frames) final;

    void statistics(const StreamStatistics& status) final;

    void eostreaming(const AudioError error) final;
};

} /* namespace audio */

} /* namespace utility */

} /* namespace adit */

#endif /* _ADIT_UTILITY_AUDIOSILENCEDETECTOR_H_ */
